unit dbConnection_u;

interface

uses
  Forms, SysUtils, Classes, DB, ADODB, StdCtrls, DBGrids, DBCtrls,
  Windows;

type
  TConnection = class(TObject)
  private

  public
    dbConnection: TADOConnection;
    dsrOne, dsrMany, dsrQRY: TDataSource;
    tblOne, tblMany: TADOTable;
    qrySolar: TADOQuery;
    procedure dbConnect;
    procedure dbDisconnect;
    procedure RestoreDatabase(var GridOne, GridMany, GridSQL: TDBGrid);
    procedure setupGrids(var GridOne, GridMany, GridSQL: TDBGrid);
    procedure DisplayAll(var QrX: TADOQuery; TableX: string);
    procedure ResetTables(GridOne, GridMany: TDBGrid);
    procedure runSQL(sSQL: string);
    procedure executeSQL(sSQL: string; GridOne, GridMany, GridSQL: TDBGrid);
    procedure ErrorMessage(sVal : String);
  end;

var
  MyForm: TForm;

implementation

{ TConnection }
uses Controls, Dialogs;

const
  DBFileName    : String = 'SolarDB.mdb';
  DBBackup      : String = 'SolarDB_Backup.mdb';
  PlanetsTable  : String = 'tblPlanets';
  MissionsTable : String = 'tblMissions';

procedure TConnection.dbConnect;
begin
  dbConnection := TADOConnection.Create(MyForm);
  dbConnection.LoginPrompt := False;
  dbConnection.ConnectionString :=
    'Provider=Microsoft.Jet.OLEDB.4.0;' + 'Data Source=' + DBFileName + ';' +
    'Mode=ReadWrite;' + 'Persist Security Info=False;' +
    'Jet OLEDB:Database Password=*************';
  dbConnection.Provider := 'Provider=Microsoft.Jet.OLEDB.4.0;';
  dbConnection.Open;

  tblOne := TADOTable.Create(MyForm);
  tblOne.Connection := dbConnection;
  tblOne.TableName  := PlanetsTable;
  tblOne.Open;
  tblOne.Sort := 'PlanetID ASC';
  tblOne.First;

  tblMany := TADOTable.Create(MyForm);
  tblMany.Connection := dbConnection;
  tblMany.TableName  := MissionsTable;
  tblMany.Open;
  tblMany.Sort := 'MissionID ASC';
  tblMany.First;

  qrySolar := TADOQuery.Create(MyForm);
  qrySolar.Connection := dbConnection;
  dsrOne := TDataSource.Create(MyForm);
  dsrOne.DataSet := tblOne;
  dsrMany := TDataSource.Create(MyForm);
  dsrMany.DataSet := tblMany;
  dsrQRY := TDataSource.Create(MyForm);
  dsrQRY.DataSet := qrySolar;
end;

procedure TConnection.dbDisconnect;
begin
  qrySolar.Free;
  qrySolar := nil;
  tblOne.Free;
  tblOne := nil;
  tblMany.Free;
  tblMany := nil;
  dbConnection.Close;
  dbConnection.Free;
  dbConnection := nil;
end;

procedure TConnection.DisplayAll(var QrX: TADOQuery; TableX: string);
begin
  QrX.Close;
  QrX.SQL.Text := Format('SELECT * FROM %s', [TableX]);
  QrX.Open;
end;

procedure TConnection.ErrorMessage(sVal : String);
begin
  MessageDLG(sVal,MTError,[MBOK],0);
end;

procedure TConnection.ResetTables(GridOne, GridMany: TDBGrid);
begin
  tblMany.Free;
  tblMany := nil;
  tblMany := TADOTable.Create(MyForm);
  tblMany.Connection := dbConnection;
  tblMany.TableName := PlanetsTable;
  tblMany.Open;
  tblMany.Sort := 'PlanetID ASC';
  tblMany.First;
  dsrMany.DataSet := tblMany;
  GridMany.datasource := dsrMany;

  tblOne.Free;
  tblOne := nil;
  tblOne := TADOTable.Create(MyForm);
  tblOne.Connection := dbConnection;
  tblOne.TableName := MissionsTable;
  tblOne.Open;
  tblOne.Sort := 'MissionID ASC';
  tblOne.First;
  dsrOne.DataSet := tblOne;
  GridOne.datasource := dsrOne;
end;

procedure TConnection.RestoreDatabase(var GridOne, GridMany, GridSQL: TDBGrid);
var
  sMsg  : string;
  bFail : boolean;
begin
  // Restore the original data in database.
  sMsg := 'Are you sure you want to reset the database?';
  if MessageDlg(sMsg, mtConfirmation, [mbYes, mbNo], 0) = mrYes then
  begin
    try
      Screen.Cursor := crHourGlass;
      try
        dbConnection.Close;
        tblOne.Destroy;
        tblMany.Destroy;
        bfail := False;
        DeleteFile(PChar(DBFileName));
        CopyFile(PChar(DBBackup), PChar(DBFileName), bfail);
        dbConnect;
        setupGrids(GridOne, GridMany, GridSQL);
        tblOne.Open;
        tblOne.Sort := 'PlanetID ASC';
        tblMany.Open;
        tblMany.Sort := 'MissionID ASC';
      except
        on E: Exception do
        begin
          Showmessage(E.Message);
        end;
      end;
    finally
      Screen.Cursor := crDefault;
    end;
    MessageDlg('Database reset complete.', mtInformation, [mbOK], 0);
  end; // if
end;

procedure TConnection.runSQL(sSQL: string);
begin
  if length(sSQL) <> 0 then
    begin
      qrySolar.Close;
      qrySolar.SQL.Text := sSQL;
      qrySolar.Open;
    end
    else
      Showmessage('No SQL statement entered');
end;

procedure TConnection.executeSQL(sSQL: string;
  GridOne, GridMany, GridSQL: TDBGrid);
var
  srefresh: string;
begin
  if length(sSQL) <> 0 then
  begin
    qrySolar.Close;
    qrySolar.SQL.Text := sSQL;
    qrySolar.ExecSQL;
    Showmessage('Database changed.');
    if pos('tblPlanets', sSQL) <> 0 then
    begin
      DisplayAll(qrySolar,'tblPlanets');
    end
    else
    begin
      DisplayAll(qrySolar,'tblMissions');
    end;
    ResetTables(GridOne, GridMany);
   end
  else
    Showmessage('No SQL statement entered');
end;

procedure TConnection.setupGrids(var GridOne, GridMany, GridSQL: TDBGrid);
begin
  GridOne.datasource := dsrOne;
  GridOne.columns[0].width := 50;
  GridOne.columns[1].width := 120;
  GridOne.columns[2].width := 50;
  GridOne.columns[3].width := 50;
  GridOne.columns[4].width := 70;
  GridOne.columns[5].width := 70;
  GridMany.datasource := dsrMany;
  GridMany.columns[0].width := 50;
  GridMany.columns[1].width := 150;
  GridMany.columns[2].width := 70;
  GridMany.columns[3].width := 50;
  GridMany.columns[4].width := 70;
  GridMany.columns[5].width := 70;
  GridSQL.datasource := dsrQRY;
end;

end.
